/**
 * Copyright (c) 2015 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Andreas Sewe - initial API and implementation.
 */
package org.eclipse.epp.internal.logging.aeri.ide.processors;

import static com.google.common.base.MoreObjects.firstNonNull;
import static com.google.common.base.Optional.*;
import static com.google.common.collect.Lists.newArrayList;
import static org.apache.commons.lang3.StringUtils.substringBeforeLast;

import java.util.Collection;
import java.util.List;
import java.util.Map.Entry;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.core.runtime.Platform;
import org.eclipse.epp.internal.logging.aeri.ide.l10n.Messages;
import org.eclipse.epp.internal.logging.aeri.ide.utils.Formats;
import org.eclipse.epp.logging.aeri.core.Constants;
import org.eclipse.epp.logging.aeri.core.IBundle;
import org.eclipse.epp.logging.aeri.core.IThrowable;
import org.eclipse.epp.logging.aeri.core.util.Logs.FakeBundle;
import org.eclipse.jdt.annotation.Nullable;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.service.packageadmin.ExportedPackage;
import org.osgi.service.packageadmin.PackageAdmin;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Optional;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;

@SuppressWarnings("deprecation")
class WiringErrorAnalyzer {

    @Nullable
    private final PackageAdmin packageAdmin;

    WiringErrorAnalyzer() {
        packageAdmin = getService(PackageAdmin.class).orNull();
    }

    private <T> Optional<T> getService(Class<T> serviceClass) {
        Bundle bundle = Platform.getBundle(Constants.BUNDLE_ID);
        if (bundle == null) {
            return absent();
        }
        BundleContext context = bundle.getBundleContext();
        if (context == null) {
            return absent();
        }

        ServiceReference<T> reference = context.getServiceReference(serviceClass);
        if (reference == null) {
            return absent();
        }
        return fromNullable(context.getService(reference));
    }

    public Optional<String> computeComment(final List<IBundle> presentBundles, IThrowable throwable) {
        if (packageAdmin == null) {
            return Optional.absent();
        }

        List<String> problematicPackages = extractProblematicPackage(throwable);
        if (problematicPackages.isEmpty()) {
            return Optional.absent();
        }

        Set<String> presentBundlesSymbolicNames = presentBundles.stream().map(bundle -> bundle.getName()).collect(Collectors.toSet());

        StringBuilder commentBuilder = new StringBuilder();

        for (String problematicPackage : problematicPackages) {
            Multimap<org.osgi.framework.Bundle, org.osgi.framework.Bundle> exportersToImporters = mapExportingToImportingBundles(
                    presentBundlesSymbolicNames, problematicPackage);
            if (!exportersToImporters.isEmpty()) {
                appendProblematicPackageComment(problematicPackage, exportersToImporters, commentBuilder);
            }
        }
        String comment = commentBuilder.toString();
        if (StringUtils.isEmpty(comment)) {
            return Optional.absent();
        } else {
            return Optional.of(comment);
        }
    }

    private Multimap<org.osgi.framework.Bundle, org.osgi.framework.Bundle> mapExportingToImportingBundles(
            Set<String> presentBundlesSymbolicNames, String problematicPackage) {
        Multimap<org.osgi.framework.Bundle, org.osgi.framework.Bundle> exportersToImporters = HashMultimap.create();
        ExportedPackage[] exportedPackages = packageAdmin.getExportedPackages(problematicPackage);
        if (ArrayUtils.isEmpty(exportedPackages)) {
            return exportersToImporters;
        }
        for (ExportedPackage exportedPackage : exportedPackages) {
            // TODO: exporting bundle will never return a fragment here. Thus, we will miss a concrete detail here.
            org.osgi.framework.Bundle exportingBundle = exportedPackage.getExportingBundle();
            if (!isPresent(exportingBundle)) {
                continue;
            }
            for (org.osgi.framework.Bundle importingBundle : exportedPackage.getImportingBundles()) {
                if (!isPresent(importingBundle)) {
                    continue;
                }
                if (presentBundlesSymbolicNames.contains(importingBundle.getSymbolicName())) {
                    exportersToImporters.put(exportingBundle, importingBundle);
                }
            }
            if (!exportersToImporters.containsKey(exportingBundle)) {
                exportersToImporters.put(exportingBundle, new FakeBundle("no bundle referenced on the stack trace."));
            }
        }
        return exportersToImporters;
    }

    private void appendProblematicPackageComment(String problematicPackage,
            Multimap<org.osgi.framework.Bundle, org.osgi.framework.Bundle> exportersToImporters, StringBuilder comment) {
        comment.append(Formats.format(Messages.COMMENT_PROBLEMATIC_PACKAGE_ORIGIN_EXPORTS, problematicPackage));
        for (Entry<org.osgi.framework.Bundle, Collection<org.osgi.framework.Bundle>> entry : exportersToImporters.asMap().entrySet()) {
            org.osgi.framework.Bundle exporter = entry.getKey();
            Collection<org.osgi.framework.Bundle> importers = entry.getValue();
            comment.append(Formats.format(Messages.COMMENT_PROBLEMATIC_PACKAGE_IMPORTS, exporter.getSymbolicName(), exporter.getVersion()));
            for (org.osgi.framework.Bundle importer : importers) {
                comment.append(Formats.format(Messages.COMMENT_PROBLEMATIC_PACKAGE_IMPORT, importer.getSymbolicName(),
                        firstNonNull(importer.getVersion(), "")));
            }
        }
    }

    private boolean isPresent(org.osgi.framework.Bundle bundle) {
        switch (bundle.getState()) {
        case org.osgi.framework.Bundle.RESOLVED:
        case org.osgi.framework.Bundle.STARTING:
        case org.osgi.framework.Bundle.ACTIVE:
            return true;
        default:
            return false;
        }
    }

    @VisibleForTesting
    public static List<String> extractProblematicPackage(IThrowable throwable) {
        String message = throwable.getMessage();
        if (StringUtils.equals(Constants.HIDDEN, message)) {
            return newArrayList();
        }
        if (NoClassDefFoundError.class.getName().equals(throwable.getClassName())
                || LinkageError.class.getName().equals(throwable.getClassName())) {
            return handleNoClassDefFoundErrorAndLinkageError(message);
        } else if (ClassNotFoundException.class.getName().equals(throwable.getClassName())) {
            return handleClassNotFoundException(message);
        } else if (NoSuchMethodError.class.getName().equals(throwable.getClassName())) {
            return handleMethodNotFoundException(message);
        } else if (VerifyError.class.getName().equals(throwable.getClassName())) {
            return handleVerifyError(message);
        } else {
            return newArrayList();
        }
    }

    private static List<String> handleVerifyError(String message) {
        List<String> packages = newArrayList();
        // extract foo/bar/baz from all patterns 'foo/bar/baz'
        Pattern pattern = Pattern.compile("'([\\S]*)'"); //$NON-NLS-1$
        Matcher matcher = pattern.matcher(message);
        while (matcher.find()) {
            // the extracted pattern package/package/Class
            String clazz = matcher.group(1);
            int lastIndexOfSlash = clazz.lastIndexOf('/');
            if (lastIndexOfSlash == -1) {
                continue;
            }
            String packageName = clazz.substring(0, lastIndexOfSlash);
            packageName = packageName.replaceAll("/", "."); //$NON-NLS-1$ //$NON-NLS-2$
            if (!packages.contains(packageName)) {
                packages.add(packageName);
            }
        }
        return packages;
    }

    private static List<String> handleNoClassDefFoundErrorAndLinkageError(String message) {
        int lastIndexOfSlash = message.lastIndexOf('/');
        if (lastIndexOfSlash < 0) {
            return newArrayList();
        } else {
            String packageName = message.substring(0, lastIndexOfSlash).replace('/', '.');
            return newArrayList(packageName);
        }
    }

    private static List<String> handleClassNotFoundException(String message) {
        int firstIndexOfSpace = message.indexOf(" "); //$NON-NLS-1$
        if (firstIndexOfSpace >= 0) {
            message = message.substring(0, firstIndexOfSpace);
        }
        int lastIndexOfDot = message.lastIndexOf('.');
        if (lastIndexOfDot < 0) {
            return newArrayList();
        } else {
            String packageName = message.substring(0, lastIndexOfDot);
            return newArrayList(packageName);
        }
    }

    private static List<String> handleMethodNotFoundException(String message) {
        // java.lang.NoSuchMethodError: org.eclipse.recommenders.utils.Checks.anyIsNull([Ljava/lang/Object;)Z
        // java.lang.NoSuchMethodError: HIDDEN
        String className = substringBeforeLast(message, "."); //$NON-NLS-1$
        // we assume that there is a package...
        String packageName = substringBeforeLast(className, "."); //$NON-NLS-1$
        return newArrayList(packageName);
    }
}
