/**
 * Copyright (c) 2010-2015, Marton Bur, Zoltan Ujhelyi, Akos Horvath, Istvan Rath and Danil Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * Contributors:
 * Marton Bur - initial API and implementation
 */
package org.eclipse.viatra.query.runtime.localsearch.planner;

import com.google.common.base.Function;
import com.google.common.collect.Sets;
import java.util.Set;
import org.eclipse.viatra.query.runtime.localsearch.planner.PConstraintCategory;
import org.eclipse.viatra.query.runtime.localsearch.planner.cost.IConstraintEvaluationContext;
import org.eclipse.viatra.query.runtime.matchers.context.IQueryBackendContext;
import org.eclipse.viatra.query.runtime.matchers.context.IQueryResultProviderAccess;
import org.eclipse.viatra.query.runtime.matchers.context.IQueryRuntimeContext;
import org.eclipse.viatra.query.runtime.matchers.psystem.PBody;
import org.eclipse.viatra.query.runtime.matchers.psystem.PConstraint;
import org.eclipse.viatra.query.runtime.matchers.psystem.PVariable;
import org.eclipse.viatra.query.runtime.matchers.psystem.analysis.QueryAnalyzer;
import org.eclipse.xtend2.lib.StringConcatenation;

/**
 * Wraps a PConstraint together with information required for the planner. Currently contains information about the expected binding state of
 * the affected variables also called application condition, and the cost of the enforcement, based on the meta and/or the runtime context.
 * 
 * @author Marton Bur
 * @noreference This class is not intended to be referenced by clients.
 */
@SuppressWarnings("all")
public class PConstraintInfo implements IConstraintEvaluationContext {
  private PConstraint constraint;
  
  private Set<PVariable> boundMaskVariables;
  
  private Set<PVariable> freeMaskVariables;
  
  private Set<PConstraintInfo> sameWithDifferentBindings;
  
  private IQueryRuntimeContext runtimeContext;
  
  private QueryAnalyzer queryAnalyzer;
  
  private IQueryResultProviderAccess resultProviderAccess;
  
  private double cost;
  
  /**
   * Instantiates the wrapper
   * @param constraintfor which the information is added and stored
   * @param boundMaskVariablesthe bound variables in the operation mask
   * @param freeMaskVariablesthe free variables in the operation mask
   * @param sameWithDifferentBindingsduring the planning process, multiple operation adornments are considered for a constraint, so that it
   * is represented by multiple plan infos. This parameter contains all plan infos that are for the same
   * constraint, but with different adornment
   * @param context the query backend context
   */
  public PConstraintInfo(final PConstraint constraint, final Set<PVariable> boundMaskVariables, final Set<PVariable> freeMaskVariables, final Set<PConstraintInfo> sameWithDifferentBindings, final IQueryBackendContext context, final Function<IConstraintEvaluationContext, Double> costFunction) {
    this.constraint = constraint;
    this.boundMaskVariables = boundMaskVariables;
    this.freeMaskVariables = freeMaskVariables;
    this.sameWithDifferentBindings = sameWithDifferentBindings;
    IQueryRuntimeContext _runtimeContext = context.getRuntimeContext();
    this.runtimeContext = _runtimeContext;
    QueryAnalyzer _queryAnalyzer = context.getQueryAnalyzer();
    this.queryAnalyzer = _queryAnalyzer;
    IQueryResultProviderAccess _resultProviderAccess = context.getResultProviderAccess();
    this.resultProviderAccess = _resultProviderAccess;
    Double _apply = costFunction.apply(this);
    this.cost = (_apply).doubleValue();
  }
  
  @Override
  public IQueryRuntimeContext getRuntimeContext() {
    return this.runtimeContext;
  }
  
  @Override
  public QueryAnalyzer getQueryAnalyzer() {
    return this.queryAnalyzer;
  }
  
  @Override
  public PConstraint getConstraint() {
    return this.constraint;
  }
  
  @Override
  public Set<PVariable> getFreeVariables() {
    return this.freeMaskVariables;
  }
  
  @Override
  public Set<PVariable> getBoundVariables() {
    return this.boundMaskVariables;
  }
  
  public Set<PConstraintInfo> getSameWithDifferentBindings() {
    return this.sameWithDifferentBindings;
  }
  
  public double getCost() {
    return this.cost;
  }
  
  public PConstraintCategory getCategory(final PBody pBody, final Set<PVariable> boundVariables) {
    Sets.SetView<PVariable> _intersection = Sets.<PVariable>intersection(this.freeMaskVariables, boundVariables);
    boolean _isEmpty = _intersection.isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      return PConstraintCategory.PAST;
    } else {
      Set<PVariable> _allVariables = pBody.getAllVariables();
      Sets.SetView<PVariable> _difference = Sets.<PVariable>difference(_allVariables, boundVariables);
      Sets.SetView<PVariable> _intersection_1 = Sets.<PVariable>intersection(this.boundMaskVariables, _difference);
      boolean _isEmpty_1 = _intersection_1.isEmpty();
      boolean _not_1 = (!_isEmpty_1);
      if (_not_1) {
        return PConstraintCategory.FUTURE;
      } else {
        return PConstraintCategory.PRESENT;
      }
    }
  }
  
  @Override
  public String toString() {
    StringConcatenation _builder = new StringConcatenation();
    String _format = String.format("\n");
    _builder.append(_format, "");
    String _string = this.constraint.toString();
    _builder.append(_string, "");
    _builder.append(", bound variables: ");
    _builder.append(this.boundMaskVariables, "");
    _builder.append(", cost: ");
    String _format_1 = String.format("%.2f", Double.valueOf(this.cost));
    _builder.append(_format_1, "");
    return _builder.toString();
  }
  
  @Override
  public IQueryResultProviderAccess resultProviderAccess() {
    return this.resultProviderAccess;
  }
}
