/**
 * Copyright (c) 2013 RCP Vision (http://www.rcp-vision.com) and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * Lorenzo Bettini - initial API and implementation
 */
package org.eclipse.emf.parsley.dsl.ui.contentassist;

import com.google.inject.Inject;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.parsley.dsl.model.ModelPackage;
import org.eclipse.emf.parsley.dsl.typing.EmfParsleyDslTypeSystem;
import org.eclipse.emf.parsley.dsl.ui.contentassist.AbstractEmfParsleyDslProposalProvider;
import org.eclipse.emf.parsley.dsl.util.EmfParsleyDslGuiceModuleHelper;
import org.eclipse.emf.parsley.dsl.util.EmfParsleyDslModelUtil;
import org.eclipse.emf.parsley.dsl.validation.EmfParsleyDslExpectedSuperTypes;
import org.eclipse.jface.text.Region;
import org.eclipse.jface.viewers.StyledString;
import org.eclipse.swt.graphics.Image;
import org.eclipse.xtext.Assignment;
import org.eclipse.xtext.common.types.JvmOperation;
import org.eclipse.xtext.common.types.JvmParameterizedTypeReference;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.common.types.JvmTypeConstraint;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.common.types.JvmWildcardTypeReference;
import org.eclipse.xtext.common.types.TypesPackage;
import org.eclipse.xtext.common.types.access.IJvmTypeProvider;
import org.eclipse.xtext.common.types.xtext.ui.ITypesProposalProvider;
import org.eclipse.xtext.resource.XtextResource;
import org.eclipse.xtext.ui.editor.contentassist.ContentAssistContext;
import org.eclipse.xtext.ui.editor.contentassist.ICompletionProposalAcceptor;
import org.eclipse.xtext.ui.editor.contentassist.PrefixMatcher;
import org.eclipse.xtext.ui.editor.model.IXtextDocument;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure2;
import org.eclipse.xtext.xbase.typesystem.references.LightweightTypeReference;
import org.eclipse.xtext.xbase.typesystem.references.LightweightTypeReferenceFactory;
import org.eclipse.xtext.xbase.ui.contentassist.ImportOrganizingProposal;
import org.eclipse.xtext.xbase.ui.contentassist.ReplacingAppendable;

/**
 * see http://www.eclipse.org/Xtext/documentation.html#contentAssist on how to customize content assistant
 */
@SuppressWarnings("all")
public class EmfParsleyDslProposalProvider extends AbstractEmfParsleyDslProposalProvider {
  @Inject
  private ITypesProposalProvider typeProposalProvider;
  
  @Inject
  private IJvmTypeProvider.Factory typeProviderFactory;
  
  @Inject
  @Extension
  private EmfParsleyDslExpectedSuperTypes _emfParsleyDslExpectedSuperTypes;
  
  @Inject
  @Extension
  private EmfParsleyDslGuiceModuleHelper _emfParsleyDslGuiceModuleHelper;
  
  @Inject
  private ReplacingAppendable.Factory appendableFactory;
  
  @Inject
  private EmfParsleyDslTypeSystem typeSystem;
  
  @Override
  public void completeViewSpecification_Type(final EObject model, final Assignment assignment, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    this.showOnlySubtypesOf(model, context, acceptor, this._emfParsleyDslExpectedSuperTypes.getExpectedSupertype(ModelPackage.Literals.VIEW_SPECIFICATION));
  }
  
  @Override
  public void completeFeatureAssociatedExpression_ParameterType(final EObject model, final Assignment assignment, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    this.showSubtypesOfEObjectForEmfFeatureAccess(model, context, acceptor);
  }
  
  @Override
  public void completeFeatureSpecification_ParameterType(final EObject model, final Assignment assignment, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    this.showSubtypesOfEObjectForEmfFeatureAccess(model, context, acceptor);
  }
  
  @Override
  public void completeControlFactorySpecification_ParameterType(final EObject model, final Assignment assignment, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    this.showSubtypesOfEObjectForEmfFeatureAccess(model, context, acceptor);
  }
  
  @Override
  public void completeExtendsClause_SuperType(final EObject model, final Assignment assignment, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    this.showOnlySubtypesOf(model, context, acceptor, this._emfParsleyDslExpectedSuperTypes.getExpectedSupertype(model.eContainer()));
  }
  
  protected void showSubtypesOfEObjectForEmfFeatureAccess(final EObject model, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    this.showOnlySubtypesOf(model, context, acceptor, this._emfParsleyDslExpectedSuperTypes.getExpectedSupertype(ModelPackage.Literals.FEATURE_ASSOCIATED_EXPRESSION));
  }
  
  protected void showOnlySubtypesOf(final EObject model, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor, final Class<?> superType) {
    if ((superType == null)) {
      return;
    }
    final IJvmTypeProvider jvmTypeProvider = this.typeProviderFactory.createTypeProvider(model.eResource().getResourceSet());
    final JvmType interfaceToImplement = jvmTypeProvider.findTypeByName(superType.getName());
    this.typeProposalProvider.createSubTypeProposals(interfaceToImplement, this, context, 
      TypesPackage.Literals.JVM_PARAMETERIZED_TYPE_REFERENCE__TYPE, acceptor);
  }
  
  @Override
  public void completeBinding_TypeDecl(final EObject model, final Assignment assignment, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    this.createStandardJavaTypesProposals(context, acceptor);
    final Procedure2<ReplacingAppendable, JvmOperation> _function = new Procedure2<ReplacingAppendable, JvmOperation>() {
      @Override
      public void apply(final ReplacingAppendable appendable, final JvmOperation op) {
        appendable.append(EmfParsleyDslProposalProvider.this.toLightweightTypeReference(op.getReturnType(), model));
        appendable.append(" ");
        appendable.append(op.getSimpleName().substring("value".length()));
      }
    };
    this.createBindingProposals(model, this._emfParsleyDslGuiceModuleHelper.getAllGuiceValueBindingsMethodsInSuperclass(EmfParsleyDslModelUtil.containingModule(model)), context, acceptor, _function);
  }
  
  @Override
  public void completeBinding_TypeToBind(final EObject model, final Assignment assignment, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    final Function1<JvmOperation, JvmTypeReference> _function = new Function1<JvmOperation, JvmTypeReference>() {
      @Override
      public JvmTypeReference apply(final JvmOperation op) {
        return EmfParsleyDslProposalProvider.this.extractWildcardUpperBound(op);
      }
    };
    this.completeTypeOrProvideBinding(model, this._emfParsleyDslGuiceModuleHelper.getAllGuiceTypeBindingsMethodsInSuperclass(EmfParsleyDslModelUtil.containingModule(model)), _function, assignment, context, acceptor);
  }
  
  @Override
  public void completeBinding_Type(final EObject model, final Assignment assignment, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    final Function1<JvmOperation, JvmTypeReference> _function = new Function1<JvmOperation, JvmTypeReference>() {
      @Override
      public JvmTypeReference apply(final JvmOperation op) {
        JvmTypeReference _xblockexpression = null;
        {
          JvmTypeReference _extractWildcardUpperBound = EmfParsleyDslProposalProvider.this.extractWildcardUpperBound(op);
          final JvmParameterizedTypeReference providerType = ((JvmParameterizedTypeReference) _extractWildcardUpperBound);
          _xblockexpression = IterableExtensions.<JvmTypeReference>head(providerType.getArguments());
        }
        return _xblockexpression;
      }
    };
    this.completeTypeOrProvideBinding(model, this._emfParsleyDslGuiceModuleHelper.getAllGuiceProviderBindingsMethodsInSuperclass(EmfParsleyDslModelUtil.containingModule(model)), _function, assignment, context, acceptor);
  }
  
  private JvmTypeReference extractWildcardUpperBound(final JvmOperation op) {
    JvmTypeReference _xblockexpression = null;
    {
      JvmTypeReference _returnType = op.getReturnType();
      final JvmParameterizedTypeReference returnType = ((JvmParameterizedTypeReference) _returnType);
      JvmTypeReference _head = IterableExtensions.<JvmTypeReference>head(returnType.getArguments());
      final JvmWildcardTypeReference argument = ((JvmWildcardTypeReference) _head);
      _xblockexpression = IterableExtensions.<JvmTypeConstraint>head(argument.getConstraints()).getTypeReference();
    }
    return _xblockexpression;
  }
  
  private void completeTypeOrProvideBinding(final EObject model, final Iterable<JvmOperation> superClassValueBindings, final Function1<? super JvmOperation, ? extends JvmTypeReference> typeExtractor, final Assignment assignment, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    this.createStandardJavaTypesProposals(context, acceptor);
    final Procedure2<ReplacingAppendable, JvmOperation> _function = new Procedure2<ReplacingAppendable, JvmOperation>() {
      @Override
      public void apply(final ReplacingAppendable appendable, final JvmOperation op) {
        final JvmTypeReference typeReference = typeExtractor.apply(op);
        appendable.append(EmfParsleyDslProposalProvider.this.toLightweightTypeReference(typeReference, model));
      }
    };
    this.createBindingProposals(model, superClassValueBindings, context, acceptor, _function);
  }
  
  /**
   * show the standard Java type completions
   */
  private void createStandardJavaTypesProposals(final ContentAssistContext context, final ICompletionProposalAcceptor acceptor) {
    this.completeJavaTypes(context, TypesPackage.Literals.JVM_PARAMETERIZED_TYPE_REFERENCE__TYPE, true, this.getQualifiedNameValueConverter(), this.createVisibilityFilter(context), acceptor);
  }
  
  private void createBindingProposals(final EObject model, final Iterable<JvmOperation> superClassValueBindings, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor, final Procedure2<? super ReplacingAppendable, ? super JvmOperation> proposalTextStrategy) {
    for (final JvmOperation op : superClassValueBindings) {
      this.createProposals(model, op, context, acceptor, proposalTextStrategy);
    }
  }
  
  private LightweightTypeReference toLightweightTypeReference(final JvmTypeReference typeRef, final EObject model) {
    return this.typeSystem.toLightweightTypeReference(typeRef, model);
  }
  
  private void createProposals(final EObject model, final JvmOperation op, final ContentAssistContext context, final ICompletionProposalAcceptor acceptor, final Procedure2<? super ReplacingAppendable, ? super JvmOperation> proposalTextStrategy) {
    final IXtextDocument document = context.getDocument();
    Resource _eResource = model.eResource();
    final XtextResource resource = ((XtextResource) _eResource);
    final int offset = context.getReplaceRegion().getOffset();
    final ReplacingAppendable appendable = this.appendableFactory.create(document, resource, offset, context.getReplaceRegion().getLength());
    proposalTextStrategy.apply(appendable, op);
    final Image image = this.getImage(op);
    final LightweightTypeReferenceFactory typeConverter = this.getTypeConverter(context.getResource());
    final StyledString displayString = this.getStyledDisplayString(op, false, 0, op.getQualifiedName(), op.getSimpleName(), typeConverter);
    final ImportOrganizingProposal completionProposal = this.createCompletionProposal(appendable, context.getReplaceRegion(), displayString, image);
    completionProposal.setPriority(1500);
    final PrefixMatcher _function = new PrefixMatcher() {
      @Override
      public boolean isCandidateMatchingPrefix(final String name, final String prefix) {
        boolean _xblockexpression = false;
        {
          final PrefixMatcher delegate = context.getMatcher();
          _xblockexpression = delegate.isCandidateMatchingPrefix(op.getSimpleName(), prefix);
        }
        return _xblockexpression;
      }
    };
    completionProposal.setMatcher(_function);
    completionProposal.setAdditionalProposalInfo(op);
    completionProposal.setHover(this.getHover());
    acceptor.accept(completionProposal);
  }
  
  private ImportOrganizingProposal createCompletionProposal(final ReplacingAppendable appendable, final Region replaceRegion, final StyledString displayString, final Image image) {
    int _offset = replaceRegion.getOffset();
    int _length = replaceRegion.getLength();
    int _offset_1 = replaceRegion.getOffset();
    return new ImportOrganizingProposal(appendable, _offset, _length, _offset_1, image, displayString);
  }
}
